/*
    EIBD eib bus access and management daemon
    Copyright (C) 2005-2006 Martin K�gler <mkoegler@auto.tuwien.ac.at>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef RULESERVER_H
#define RULESERVER_H

#include <list>
#include <string>
#include "config.h"
#include "objectcontroller.h"

class Condition
{
public:
    virtual ~Condition() {};

    static Condition* create(const std::string& type, ChangeListener* cl);
    static Condition* create(ticpp::Element* pConfig, ChangeListener* cl);

    virtual bool evaluate() = 0;
    virtual void importXml(ticpp::Element* pConfig) = 0;
    virtual void exportXml(ticpp::Element* pConfig) = 0;

    typedef std::list<Condition*> ConditionsList_t;
};

class AndCondition : public Condition
{
public:
    AndCondition(ChangeListener* cl);
    virtual ~AndCondition();

    virtual bool evaluate();
    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);
    
private:
    ConditionsList_t conditionsList_m;
    ChangeListener* cl_m;
};

class OrCondition : public Condition
{
public:
    OrCondition(ChangeListener* cl);
    virtual ~OrCondition();

    virtual bool evaluate();
    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);
    
private:
    ConditionsList_t conditionsList_m;
    ChangeListener* cl_m;
};

class NotCondition : public Condition
{
public:
    NotCondition(ChangeListener* cl);
    virtual ~NotCondition();

    virtual bool evaluate();
    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);
    
private:
    Condition* condition_m;
    ChangeListener* cl_m;
};

class ObjectCondition : public Condition
{
public:
    ObjectCondition(ChangeListener* cl);
    virtual ~ObjectCondition();

    virtual bool evaluate();
    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);
    
private:
    Object* object_m;
    std::string type_m;
    std::string value_m;
    ChangeListener* cl_m;
};

class TimerCondition : public Condition
{
public:
    TimerCondition(ChangeListener* cl);
    virtual ~TimerCondition();

    virtual bool evaluate();
    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);
    
private:
    std::string at_m;
    ChangeListener* cl_m;
};

class Action : protected Thread
{
public:
    Action() : delay_m(0) {};
    virtual ~Action() {};

    static Action* create(ticpp::Element* pConfig);
    static Action* create(const std::string& type);

    virtual void importXml(ticpp::Element* pConfig) = 0;
    virtual void exportXml(ticpp::Element* pConfig) = 0;

    virtual void execute() { Start(); };
private:
    virtual void Run (pth_sem_t * stop) = 0;
protected:
    int delay_m;
};

class DimUpAction : public Action
{
public:
    DimUpAction();
    virtual ~DimUpAction();

    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);

private:
    virtual void Run (pth_sem_t * stop);

    Object* object_m;
    int start_m, stop_m, duration_m;
};

class SetValueAction : public Action
{
public:
    SetValueAction();
    virtual ~SetValueAction();

    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);

private:
    virtual void Run (pth_sem_t * stop);

    Object* object_m;
    float value_m;
};

class Rule : public ChangeListener
{
public:
    Rule();
    virtual ~Rule();

    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);

    virtual void onChange(Object* object);

    void evaluate();
    
private:
    Condition* condition_m;
    typedef std::list<Action*> ActionsList_t;
    ActionsList_t actionsList_m;
};

class RuleServer
{
public:
    RuleServer();
    ~RuleServer();

    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);

private:
    std::list<Rule*> rulesList_m;
};

#endif
