/*
    Linknx high level management application
    Copyright (C) 2007 Jean-François Meessen <jf.meessen@ouaye.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef OBJECTCONTROLLER_H
#define OBJECTCONTROLLER_H

#include <list>
#include <string>
#include <map>
#define TIXML_USE_TICPP
#include "ticpp.h"
#include "eibclient.h"

class Object;

class ChangeListener
{
public:
    virtual void onChange(Object* object) = 0;
};

class Object
{
public:
    Object();
    virtual ~Object();

    const char* getID() { return id_m.c_str(); };
    void onUpdate();
    void addChangeListener(ChangeListener* listener);
    void removeChangeListener(ChangeListener* listener);
    virtual void setValue(int value) {};
    virtual void setValue(float value) {};
    virtual void setValue(bool value) {};
    virtual int getIntValue() { return getBoolValue() ? 0 : 1; };
    virtual float getFloatValue() { return getBoolValue() ? 0 : 1; };
    virtual bool getBoolValue() { return false; };
private:
    std::string id_m;
    eibaddr_t gad_m;
    std::list<ChangeListener*> listenerList_m;
};

class SwitchingObject : public Object
{
public:
    SwitchingObject();
    virtual ~SwitchingObject();

    virtual void setIntValue(int value);
    virtual void setFloatValue(float value);
    virtual void setBoolValue(bool value);
    virtual int getIntValue() { return getBoolValue() ? 0 : 1; };
    virtual float getFloatValue() { return getBoolValue() ? 0 : 1; };
    virtual bool getBoolValue() { return value_m; };
private:
    bool value_m;
};

class ValueObject : public Object
{
public:
    ValueObject();
    virtual ~ValueObject();

    virtual void setIntValue(int value);
    virtual void setFloatValue(float value);
    virtual void setBoolValue(bool value);
    virtual int getIntValue() { return (int)getFloatValue(); };
    virtual float getFloatValue() { return value_m; };
    virtual bool getBoolValue() { return getFloatValue() != 0; };
private:
    float value_m;
};

class ScalingObject : public Object
{
public:
    ScalingObject();
    virtual ~ScalingObject();

    virtual void setIntValue(int value);
    virtual void setFloatValue(float value);
    virtual void setBoolValue(bool value);
    virtual int getIntValue() { return value_m; };
    virtual float getFloatValue() { return getIntValue(); };
    virtual bool getBoolValue() { return getIntValue(); };
private:
    int value_m;
};

class ObjectController
{
public:
    static ObjectController* instance();
    void addObject(Object* object);
    void removeObject(Object* object);

    Object* getObject(const std::string& id);

    virtual void importXml(ticpp::Element* pConfig);
    virtual void exportXml(ticpp::Element* pConfig);

private:
    ObjectController();
    virtual ~ObjectController();
    std::map<eibaddr_t ,Object*> objectMap_m;
    std::map<std::string ,Object*> objectIdMap_m;
    static ObjectController* instance_m;
};

#endif
