/*
    LinKNX KNX home automation platform
    Copyright (C) 2007 Jean-François Meessen <linknx@ouaye.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "ruleserver.h"

RuleServer* RuleServer::instance_m;

RuleServer::RuleServer()
{
}

RuleServer::~RuleServer()
{
  std::list<Rule*>::iterator it;
  for (it = rulesList_m.begin(); it != rulesList_m.end(); it++)
    delete (*it);
}

RuleServer* RuleServer::instance()
{
  if (instance_m == 0)
    instance_m = new RuleServer();
  return instance_m;
}

void RuleServer::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child;
  for ( child = pConfig->FirstChildElement(); child != child.end(); child++ )
  {
    Rule* rule = new Rule();
    rule->importXml(&(*child));
    rulesList_m.push_back(rule);
  }
}

void RuleServer::exportXml(ticpp::Element* pConfig)
{
  std::list<Rule*>::iterator it;
  for (it = rulesList_m.begin(); it != rulesList_m.end(); it++)
  {
    ticpp::Element* pElem = new ticpp::Element("rule");
    (*it)->exportXml(pElem);
    pConfig->LinkEndChild(pElem);
  }
}

Rule::Rule() : condition_m(0), prevValue_m(false), isActive_m(false)
{
}

Rule::~Rule()
{
  if (condition_m != 0)
    delete condition_m;

  ActionsList_t::iterator it;
  for(it=actionsList_m.begin(); it != actionsList_m.end(); ++it)
    delete (*it);
  for(it=actionsListFalse_m.begin(); it != actionsListFalse_m.end(); ++it)
    delete (*it);
}

void Rule::importXml(ticpp::Element* pConfig)
{
  pConfig->GetAttribute("id", &id_m, false);
//  pConfig->GetAttributeOrDefault("active", &isActive_m, true);

  std::string value = pConfig->GetAttribute("active");
  isActive_m = !(value == "off" || value == "false" || value == "no");

  std::cout << "Rule: Configuring " << getID() << " (active=" << isActive_m << ")" << std::endl;

  ticpp::Element* pCondition = pConfig->FirstChildElement("condition");
  condition_m = Condition::create(pCondition, this);

  ticpp::Iterator<ticpp::Element> actionListIt("actionlist");
  for ( actionListIt = pConfig->FirstChildElement("actionlist"); actionListIt != actionListIt.end(); actionListIt++ )
  {
    bool isFalse = (*actionListIt).GetAttribute("type") == "on-false";
    std::cout << "ActionList: Configuring " << (isFalse ? "'on-false'" : "" ) << std::endl;
		ticpp::Iterator<ticpp::Element> actionIt("action");
		for (actionIt = (*actionListIt).FirstChildElement("action"); actionIt != actionIt.end(); actionIt++ )
		{
			Action* action = Action::create(&(*actionIt));
      if (isFalse)
			  actionsListFalse_m.push_back(action);
      else
			  actionsList_m.push_back(action);
		}
  }
  std::cout << "Rule: Configuration done" << std::endl;
}

void Rule::exportXml(ticpp::Element* pConfig)
{
    if (id_m != "")
        pConfig->SetAttribute("id", id_m);
    if (isActive_m == false)
        pConfig->SetAttribute("active", "no");
    if (condition_m)
    {
        ticpp::Element* pCond = new ticpp::Element("condition");
        condition_m->exportXml(pCond);
        pConfig->LinkEndChild(pCond);
    }

    ActionsList_t::iterator it;
    if (actionsList_m.begin() != actionsList_m.end())
    {
        ticpp::Element* pList = new ticpp::Element("actionlist");
        pConfig->LinkEndChild(pList);
    
        for(it=actionsList_m.begin(); it != actionsList_m.end(); ++it)
        {
            ticpp::Element* pElem = new ticpp::Element("action");
            (*it)->exportXml(pElem);
            pList->LinkEndChild(pElem);
        }
    }
    if (actionsListFalse_m.begin() != actionsListFalse_m.end())
    {
        ticpp::Element* pList = new ticpp::Element("actionlist");
        pList->SetAttribute("type", "on-false");
        pConfig->LinkEndChild(pList);
    
        for(it=actionsListFalse_m.begin(); it != actionsListFalse_m.end(); ++it)
        {
            ticpp::Element* pElem = new ticpp::Element("action");
            (*it)->exportXml(pElem);
            pList->LinkEndChild(pElem);
        }
    }
}

void Rule::onChange(Object* object)
{
  evaluate();
}

void Rule::evaluate()
{
  if (isActive_m)
  {
    ActionsList_t::iterator it;
    bool curValue = condition_m->evaluate();
    if (curValue && !prevValue_m)
    {
      for(it=actionsList_m.begin(); it != actionsList_m.end(); ++it)
        (*it)->execute();
    }
    else if (!curValue && prevValue_m)
    {
      for(it=actionsListFalse_m.begin(); it != actionsListFalse_m.end(); ++it)
        (*it)->execute();
    }
    prevValue_m = curValue;
  }
}

Action* Action::create(const std::string& type)
{
  if (type == "dim-up")
    return new DimUpAction();
  else if (type == "set-value")
    return new SetValueAction();
//  else if (type == "switch")
//    return new SwitchAction();
//  else if (type == "sms")
//    return new SMSAction();
  else
    return 0;
}

Action* Action::create(ticpp::Element* pConfig)
{
  std::string type = pConfig->GetAttribute("type");
  int delay;
  pConfig->GetAttributeOrDefault("delay", &delay, 0);
  Action* action = Action::create(type);
  if (action == 0)
  {
		std::stringstream msg;
		msg << "Action type not supported: '" << type << "'" << std::endl;
    throw ticpp::Exception(msg.str());
  }
  action->delay_m = delay;
  action->importXml(pConfig);
  return action;
}

void Action::exportXml(ticpp::Element* pConfig)
{
  if (delay_m != 0)
    pConfig->SetAttribute("delay", delay_m);
}

DimUpAction::DimUpAction() : start_m(0), stop_m(255), duration_m(60), object_m(0)
{
}

DimUpAction::~DimUpAction()
{
}

void DimUpAction::importXml(ticpp::Element* pConfig)
{
    std::string id;
    id = pConfig->GetAttribute("id");
    object_m = ObjectController::instance()->getObject(id);
    pConfig->GetAttribute("start", &start_m);
    pConfig->GetAttribute("stop", &stop_m);
    pConfig->GetAttribute("duration", &duration_m);
    std::cout << "DimUpAction: Configured for object " << object_m->getID()
              << " with start=" << start_m
              << "; stop=" << stop_m
              << "; duration=" << duration_m << std::endl;
}

void DimUpAction::exportXml(ticpp::Element* pConfig)
{
    pConfig->SetAttribute("type", "dim-up");
    pConfig->SetAttribute("id", object_m->getID());
    pConfig->SetAttribute("start", start_m);
    pConfig->SetAttribute("stop", stop_m);
    pConfig->SetAttribute("duration", duration_m);

    Action::exportXml(pConfig);
}

void DimUpAction::Run (pth_sem_t * stop)
{
    pth_sleep(delay_m);
    std::cout << "Execute DimUpAction" << std::endl;

		unsigned long step = duration_m * 8333;
	  for (int idx=start_m; idx < stop_m; idx++) {
        object_m->setIntValue(idx);
        pth_usleep(step);
        if (object_m->getIntValue() < idx) {
            std::cout << "Abort DimUpAction" << std::endl;
            return;
        }
    }
}

SetValueAction::SetValueAction() : value_m(0), object_m(0)
{
}

SetValueAction::~SetValueAction()
{
}

void SetValueAction::importXml(ticpp::Element* pConfig)
{
    std::string id;
    id = pConfig->GetAttribute("id");
    object_m = ObjectController::instance()->getObject(id);

    std::string value;
    value = pConfig->GetAttribute("value");

    std::istringstream val(value);
    val >> value_m;

    if ( val.fail() )
    {
				if (value == "on" || value == "true" || value == "comfort")
						value_m = 1;
				else if (value == "off" || value == "false")
						value_m = 0;
				else if (value == "standby")
						value_m = 2;
				else if (value == "night")
						value_m = 3;
				else if (value == "frost")
						value_m = 4;
        else
        {
            std::stringstream msg;
            msg << "SetValueAction: Bad value: '" << value << "'" << std::endl;
            throw ticpp::Exception(msg.str());
        }
    }
    std::cout << "SetValueAction: Configured for object " << object_m->getID() << " with value " << value_m << std::endl;
}

void SetValueAction::exportXml(ticpp::Element* pConfig)
{
    pConfig->SetAttribute("type", "set-value");
    pConfig->SetAttribute("id", object_m->getID());
    pConfig->SetAttribute("value", value_m);

    Action::exportXml(pConfig);
}

void SetValueAction::Run (pth_sem_t * stop)
{
    pth_sleep(delay_m);
    std::cout << "Execute SetValueAction with value " << value_m << std::endl;
    if (object_m)
        object_m->setFloatValue(value_m);
}

Condition* Condition::create(const std::string& type, ChangeListener* cl)
{
  if (type == "and")
    return new AndCondition(cl);
  else if (type == "or")
    return new OrCondition(cl);
  else if (type == "not")
    return new NotCondition(cl);
  else if (type == "object")
    return new ObjectCondition(cl);
  else if (type == "timer")
    return new TimerCondition(cl);
  else
    return 0;
}

Condition* Condition::create(ticpp::Element* pConfig, ChangeListener* cl)
{
  std::string type;
  type = pConfig->GetAttribute("type");
  Condition* condition = Condition::create(type, cl);
  if (condition == 0)
  {
		std::stringstream msg;
		msg << "Condition type not supported: '" << type << "'";
    throw ticpp::Exception(msg.str());
  }
  condition->importXml(pConfig);
  return condition;
}

AndCondition::AndCondition(ChangeListener* cl) : cl_m(cl)
{
}

AndCondition::~AndCondition()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    delete (*it);
}

bool AndCondition::evaluate()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    if (!(*it)->evaluate())
      return false;
  return true;
}

void AndCondition::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child("condition");
  for ( child = pConfig->FirstChildElement("condition"); child != child.end(); child++ )
  {
    Condition* condition = Condition::create(&(*child), cl_m);
    conditionsList_m.push_back(condition);
  }
}

void AndCondition::exportXml(ticpp::Element* pConfig)
{
  pConfig->SetAttribute("type", "and");
  ConditionsList_t::iterator it;
  for (it = conditionsList_m.begin(); it != conditionsList_m.end(); it++)
  {
    ticpp::Element* pElem = new ticpp::Element("condition");
    (*it)->exportXml(pElem);
    pConfig->LinkEndChild(pElem);
  }
}
  
OrCondition::OrCondition(ChangeListener* cl) : cl_m(cl)
{
}

OrCondition::~OrCondition()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    delete (*it);
}

bool OrCondition::evaluate()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    if ((*it)->evaluate())
      return true;
  return false;
}

void OrCondition::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child("condition");
  for ( child = pConfig->FirstChildElement("condition"); child != child.end(); child++ )
  {
    Condition* condition = Condition::create(&(*child), cl_m);
    conditionsList_m.push_back(condition);
  }
}

void OrCondition::exportXml(ticpp::Element* pConfig)
{
  pConfig->SetAttribute("type", "or");
  ConditionsList_t::iterator it;
  for (it = conditionsList_m.begin(); it != conditionsList_m.end(); it++)
  {
    ticpp::Element* pElem = new ticpp::Element("condition");
    (*it)->exportXml(pElem);
    pConfig->LinkEndChild(pElem);
  }
}
  
NotCondition::NotCondition(ChangeListener* cl) : cl_m(cl), condition_m(0)
{
}

NotCondition::~NotCondition()
{
  if (condition_m)
    delete condition_m;
}

bool NotCondition::evaluate()
{
  return !condition_m->evaluate();
}

void NotCondition::importXml(ticpp::Element* pConfig)
{
  condition_m = Condition::create(pConfig->FirstChildElement("condition"), cl_m);
}

void NotCondition::exportXml(ticpp::Element* pConfig)
{
  pConfig->SetAttribute("type", "not");
  if (condition_m)
  {
    ticpp::Element* pElem = new ticpp::Element("condition");
    condition_m->exportXml(pElem);
    pConfig->LinkEndChild(pElem);
  }
}
  
ObjectCondition::ObjectCondition(ChangeListener* cl) : cl_m(cl), trigger_m(false)
{
}

ObjectCondition::~ObjectCondition()
{
}

bool ObjectCondition::evaluate()
{
    bool val = object_m->getIntValue() == value_m;
    std::cout << "ObjectCondition evaluated as '" << val << "' value_m='" << value_m << "' getIntValue='" << object_m->getIntValue() << "'" << std::endl;
    return val;
}

void ObjectCondition::importXml(ticpp::Element* pConfig)
{
    std::string trigger;
    trigger = pConfig->GetAttribute("trigger");
    std::string id;
    id = pConfig->GetAttribute("id");
    object_m = ObjectController::instance()->getObject(id);

    if (trigger == "true")
    {
        trigger_m = true;
        object_m->addChangeListener(cl_m);
    }

    std::string value;
    value = pConfig->GetAttribute("value");

    std::istringstream val(value);
    val >> value_m;

    if ( val.fail() )
    {
				if (value == "on" || value == "true" || value == "comfort")
						value_m = 1;
				else if (value == "off" || value == "false")
						value_m = 0;
				else if (value == "standby")
						value_m = 2;
				else if (value == "night")
						value_m = 3;
				else if (value == "frost")
						value_m = 4;
        else
        {
            std::stringstream msg;
            msg << "ObjectCondition: Bad value: '" << value << "'" << std::endl;
            throw ticpp::Exception(msg.str());
        }
    }
    std::cout << "ObjectCondition: configured value_m='" << value_m << "' (rdstate=" << (int)val.rdstate() << "; bad=" << val.bad() << "; good="<< val.good() << ")"<< std::endl;
}

void ObjectCondition::exportXml(ticpp::Element* pConfig)
{
  pConfig->SetAttribute("type", "object");
  pConfig->SetAttribute("id", object_m->getID());
  pConfig->SetAttribute("value", value_m);
  if (trigger_m)
    pConfig->SetAttribute("trigger", "true");
}
  

TimerCondition::TimerCondition(ChangeListener* cl)
    : PeriodicTask(cl), trigger_m(false)
{
}

TimerCondition::~TimerCondition()
{
}

bool TimerCondition::evaluate()
{
  std::cout << "TimerCondition evaluated as '" << value_m << "'" << std::endl;
  return value_m;
}

void TimerCondition::importXml(ticpp::Element* pConfig)
{
    std::string trigger;
    trigger = pConfig->GetAttribute("trigger");

    if (trigger == "true")
        trigger_m = true;
    else
        cl_m = 0;

    ticpp::Element* at = pConfig->FirstChildElement("at");
    at_m.importXml(at);
    
    ticpp::Element* during = pConfig->FirstChildElement("during", false);
    ticpp::Element* until = pConfig->FirstChildElement("until", false);
    if (during && until)
        throw ticpp::Exception("Timer can't define <until> and <during> elements simultaneously");
    if (during)
        during->GetText(&during_m);
    else if (until)
    {
        until_m.importXml(until);
        during_m = -1;
    }
    else
        during_m = 0;

    reschedule(0);
}

void TimerCondition::exportXml(ticpp::Element* pConfig)
{
    pConfig->SetAttribute("type", "timer");
    if (trigger_m)
        pConfig->SetAttribute("trigger", "true");

    ticpp::Element* pAt = new ticpp::Element("at");
    at_m.exportXml(pAt);
    pConfig->LinkEndChild(pAt);
    
    if (during_m == -1)
    {
        ticpp::Element* pUntil = new ticpp::Element("until");
        until_m.exportXml(pUntil);
        pConfig->LinkEndChild(pUntil);
    }
    else if (during_m != 0)
    {
        ticpp::Element* pDuring = new ticpp::Element("during");
        pDuring->SetText(during_m);
        pConfig->LinkEndChild(pDuring);
    }
}
  
