/*
    EIBD eib bus access and management daemon
    Copyright (C) 2005-2006 Martin K�gler <mkoegler@auto.tuwien.ac.at>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "ruleserver.h"

RuleServer::RuleServer()
{
}

RuleServer::~RuleServer()
{
}

void RuleServer::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child;
  for ( child = pConfig->FirstChildElement(); child != child.end(); child++ )
  {
    Rule* rule = new Rule();
    rule->importXml(&(*child));
    rulesList_m.push_back(rule);
  }
}

void RuleServer::exportXml(ticpp::Element* pConfig)
{

}

Rule::Rule() : condition_m(0)
{
}

Rule::~Rule()
{
  if (condition_m != 0)
    delete condition_m;

  ActionsList_t::iterator it;
  for(it=actionsList_m.begin(); it != actionsList_m.end(); ++it)
    delete (*it);
}

void Rule::importXml(ticpp::Element* pConfig)
{
  ticpp::Element* pCondition = pConfig->FirstChildElement("condition");
  condition_m = Condition::create(pCondition, this);

  ticpp::Iterator< ticpp::Element > child("action");
  for ( child = pConfig->FirstChildElement("action"); child != child.end(); child++ )
  {
    Action* action = Action::create(&(*child));
    actionsList_m.push_back(action);
  }
}

void Rule::exportXml(ticpp::Element* pConfig)
{
}

void Rule::onChange(Object* object)
{
  evaluate();
}

void Rule::evaluate()
{
  ActionsList_t::iterator it;
  if (condition_m->evaluate())
  {
    for(it=actionsList_m.begin(); it != actionsList_m.end(); ++it)
      (*it)->execute();
  }
}

Action* Action::create(const std::string& type)
{
  if (type == "dim-up")
    return new DimUpAction();
  else if (type == "set-value")
    return new SetValueAction();
//  else if (type == "switch")
//    return new SwitchAction();
//  else if (type == "sms")
//    return new SMSAction();
  else
    return 0;
}

Action* Action::create(ticpp::Element* pConfig)
{
  std::string type = pConfig->GetAttribute("type");
  int delay;
  pConfig->GetAttributeOrDefault("delay", &delay, 0);
  Action* action = Action::create(type);
  if (action == 0)
  {
		std::stringstream msg;
		msg << "Action type not supported: '" << type << "'" << std::endl;
    throw ticpp::Exception(msg.str());
  }
  action->delay_m = delay;
  action->importXml(pConfig);
  return action;
}

DimUpAction::DimUpAction() : start_m(0), stop_m(255), duration_m(60), object_m(0)
{
}

DimUpAction::~DimUpAction()
{
}

void DimUpAction::importXml(ticpp::Element* pConfig)
{
    std::string id;
    id = pConfig->FirstChildElement("object")->GetAttribute("id");
    object_m = ObjectController::instance()->getObject(id);
    pConfig->FirstChildElement("start")->GetText(start_m);
    pConfig->FirstChildElement("stop")->GetText(stop_m);
    pConfig->FirstChildElement("duration")->GetText(duration_m);
}

void DimUpAction::exportXml(ticpp::Element* pConfig)
{
}

void DimUpAction::Run (pth_sem_t * stop)
{
    pth_sleep(delay_m);
    std::cout << "Execute DimUpAction" << std::endl;

		unsigned long step = duration_m * 8333;
	  for (int idx=start_m; idx < stop_m; idx++) {
        object_m->setIntValue(idx);
        pth_usleep(step);
        if (object_m->getIntValue() < idx) {
            std::cout << "Abort DimUpAction" << std::endl;
            return;
        }
    }
}

SetValueAction::SetValueAction() : value_m(0), object_m(0)
{
}

SetValueAction::~SetValueAction()
{
}

void SetValueAction::importXml(ticpp::Element* pConfig)
{
    std::string id;
    id = pConfig->FirstChildElement("object")->GetAttribute("id");
    object_m = ObjectController::instance()->getObject(id);
    ticpp::Element* value = pConfig->FirstChildElement("value");
//    std::string type = value->GetAttribute("type");
//    if (type == "float")
//    else
    value->GetText(&value_m);
    std::cout << "Configured SetValueAction for object " << object_m->getID() << " with value " << value_m << std::endl;
}

void SetValueAction::exportXml(ticpp::Element* pConfig)
{
}

void SetValueAction::Run (pth_sem_t * stop)
{
    pth_sleep(delay_m);
    std::cout << "Execute SetValueAction with value " << value_m << std::endl;
    if (object_m)
        object_m->setFloatValue(value_m);
}

Condition* Condition::create(const std::string& type, ChangeListener* cl)
{
  if (type == "and")
    return new AndCondition(cl);
  else if (type == "or")
    return new OrCondition(cl);
  else if (type == "not")
    return new NotCondition(cl);
  else if (type == "object")
    return new ObjectCondition(cl);
  else if (type == "timer")
    return new TimerCondition(cl);
  else
    return 0;
}

Condition* Condition::create(ticpp::Element* pConfig, ChangeListener* cl)
{
  std::string type;
  type = pConfig->GetAttribute("type");
  Condition* condition = Condition::create(type, cl);
  if (condition == 0)
  {
		std::stringstream msg;
		msg << "Condition type not supported: '" << type << "'";
    throw ticpp::Exception(msg.str());
  }
  condition->importXml(pConfig);
  return condition;
}

AndCondition::AndCondition(ChangeListener* cl) : cl_m(cl)
{
}

AndCondition::~AndCondition()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    delete (*it);
}

bool AndCondition::evaluate()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    if (!(*it)->evaluate())
      return false;
  return true;
}

void AndCondition::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child("condition");
  for ( child = pConfig->FirstChildElement("condition"); child != child.end(); child++ )
  {
    Condition* condition = Condition::create(&(*child), cl_m);
    conditionsList_m.push_back(condition);
  }
}

void AndCondition::exportXml(ticpp::Element* pConfig)
{
}
  
OrCondition::OrCondition(ChangeListener* cl) : cl_m(cl)
{
}

OrCondition::~OrCondition()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    delete (*it);
}

bool OrCondition::evaluate()
{
  ConditionsList_t::iterator it;
  for(it=conditionsList_m.begin(); it != conditionsList_m.end(); ++it)
    if ((*it)->evaluate())
      return true;
  return false;
}

void OrCondition::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child("condition");
  for ( child = pConfig->FirstChildElement("condition"); child != child.end(); child++ )
  {
    Condition* condition = Condition::create(&(*child), cl_m);
    conditionsList_m.push_back(condition);
  }
}

void OrCondition::exportXml(ticpp::Element* pConfig)
{
}
  
NotCondition::NotCondition(ChangeListener* cl) : cl_m(cl), condition_m(0)
{
}

NotCondition::~NotCondition()
{
  if (condition_m)
    delete condition_m;
}

bool NotCondition::evaluate()
{
  return !condition_m->evaluate();
}

void NotCondition::importXml(ticpp::Element* pConfig)
{
  condition_m = Condition::create(pConfig->FirstChildElement("condition"), cl_m);
}

void NotCondition::exportXml(ticpp::Element* pConfig)
{
}
  
ObjectCondition::ObjectCondition(ChangeListener* cl) : cl_m(cl)
{
}

ObjectCondition::~ObjectCondition()
{
}

bool ObjectCondition::evaluate()
{
  if (type_m == "bool")
    return object_m->getBoolValue() == (value_m == "true");
//  if (type_m == "int")
//    return object_m->getIntValue() == atoi(value_m);
  return false;
}

void ObjectCondition::importXml(ticpp::Element* pConfig)
{
    std::string trigger;
    trigger = pConfig->GetAttribute("trigger");
    std::string id;
    id = pConfig->FirstChildElement("object")->GetAttribute("id");
    object_m = ObjectController::instance()->getObject(id);

    if (trigger == "true")
        object_m->addChangeListener(cl_m);

    ticpp::Element* pValue = pConfig->FirstChildElement("value");
    type_m = pValue->GetAttributeOrDefault("type", "bool");
    value_m = pValue->GetText();
}

void ObjectCondition::exportXml(ticpp::Element* pConfig)
{
}
  

TimerCondition::TimerCondition(ChangeListener* cl)
    : cl_m(cl), nextExecTime_m(0), value_m(false), during_m(0)
{
}

TimerCondition::~TimerCondition()
{
}

bool TimerCondition::evaluate()
{
  std::cout << "TimerCondition evaluated as '" << value_m << "'" << std::endl;
  return value_m;
}

void TimerCondition::importXml(ticpp::Element* pConfig)
{
    std::string trigger;
    trigger = pConfig->GetAttribute("trigger");

    ticpp::Element* at = pConfig->FirstChildElement("at");
    at->GetAttributeOrDefault("month", &(at_m.tm_mon), -1);
    at->GetAttributeOrDefault("day", &(at_m.tm_mday), -1);
    at->GetAttributeOrDefault("hour", &(at_m.tm_hour), -1);
    at->GetAttributeOrDefault("min", &(at_m.tm_min), -1);

    ticpp::Element* during = pConfig->FirstChildElement("during", false);
    ticpp::Element* until = pConfig->FirstChildElement("until", false);
    if (during && until)
        throw ticpp::Exception("Timer can't define <until> and <during> elements simultaneously");
    if (during)
        during->GetText(&during_m);
    else if (until)
    {
        until->GetAttributeOrDefault("month", &(until_m.tm_mon), -1);
        until->GetAttributeOrDefault("day", &(until_m.tm_mday), -1);
        until->GetAttributeOrDefault("hour", &(until_m.tm_hour), -1);
        until->GetAttributeOrDefault("min", &(until_m.tm_min), -1);
        during_m = -1;
    }
    else
        during_m = 0;

    reschedule(0);
}

void TimerCondition::exportXml(ticpp::Element* pConfig)
{
}
  
void TimerCondition::onTimer(time_t time)
{
    value_m = !value_m;
    if (cl_m)
        cl_m->onChange(0);
    if (during_m == 0)
        value_m = false;
}
  
void TimerCondition::reschedule(time_t now)
{
    struct tm* next;
    struct tm * timeinfo;
    if (now == 0)
        now = time(0);
    if (value_m && during_m > 0)
    {
        nextExecTime_m += during_m;
        timeinfo = localtime(&nextExecTime_m);
    }
    else
    {
				if (value_m && during_m == -1)
						next = &until_m;
				else
						next = &at_m;
		
				timeinfo = localtime(&now);
				timeinfo->tm_min++;
				if (next->tm_min != -1)
				{
						if  (timeinfo->tm_min > next->tm_min)
								timeinfo->tm_hour++;
						timeinfo->tm_min = next->tm_min;
				}
				if (next->tm_hour != -1)
				{
						if (timeinfo->tm_hour > next->tm_hour)
								timeinfo->tm_mday++;
						timeinfo->tm_hour = next->tm_hour;
				}
				if (next->tm_mday != -1)
				{
						if (timeinfo->tm_mday > next->tm_mday)
								timeinfo->tm_mon++;
						timeinfo->tm_mday = next->tm_mday;
				}
				if (next->tm_mon != -1)
				{
						if (timeinfo->tm_mon > next->tm_mon)
								timeinfo->tm_year++;
						timeinfo->tm_mon = next->tm_mon;
				}
		
				timeinfo->tm_sec = 0;
				nextExecTime_m = mktime(timeinfo);
    }
    std::cout << "TimerCondition rescheduled at "
              << timeinfo->tm_year + 1900 << "-" 
              << timeinfo->tm_mon + 1 << "-"
              << timeinfo->tm_mday << " "
              << timeinfo->tm_hour << ":"
              << timeinfo->tm_min << ":0 ("
              << nextExecTime_m << ")" << std::endl;
    TimerManager::instance()->addTask(this);
}
