/*
    LinKNX KNX home automation platform
    Copyright (C) 2007 Jean-François Meessen <linknx@ouaye.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "objectcontroller.h"
#include "persistentstorage.h"
#include "services.h"
extern "C" {
#include "common.h"
}

ObjectController* ObjectController::instance_m;

Object::Object() : gad_m(0), init_m(false), readPending_m(false)
{
}

Object::~Object()
{
}

Object* Object::create(const std::string& type)
{
  if (type == "EIS5")
    return new ValueObject();
  else if (type == "EIS6")
    return new ScalingObject();
  else
    return new SwitchingObject();
}

Object* Object::create(ticpp::Element* pConfig)
{
  std::string type = pConfig->GetAttribute("type");
  Object* obj = Object::create(type);
  if (obj == 0)
  {
		std::stringstream msg;
		msg << "Object type not supported: '" << type << "'" << std::endl;
    throw ticpp::Exception(msg.str());
  }
  obj->importXml(pConfig);
  return obj;
}

void Object::importXml(ticpp::Element* pConfig)
{
  id_m = pConfig->GetAttribute("id");
  std::string gad = pConfig->GetAttribute("gad");
  if (id_m == "")
    throw ticpp::Exception("Missing or empty object ID");
  ObjectController::instance()->addObject(this);

  if (gad != "")
    gad_m = readgaddr(gad.c_str());

  initValue_m = pConfig->GetAttribute("init");
  if (initValue_m != "" && initValue_m != "request") {
		std::istringstream val(initValue_m);
		float init_value;
		val >> init_value;
	
		if ( val.fail() )
		{
			if (initValue_m == "persist")
				init_value = PersistentStorage::read(id_m);
			else if (initValue_m == "on" || initValue_m == "true" || initValue_m == "comfort")
				init_value = 1;
			else if (initValue_m == "off" || initValue_m == "false")
				init_value = 0;
			else if (initValue_m == "standby")
				init_value = 2;
			else if (initValue_m == "night")
				init_value = 3;
			else if (initValue_m == "frost")
				init_value = 4;
			else
			{
				std::stringstream msg;
				msg << "Object: Bad init value: '" << initValue_m << "'" << std::endl;
				throw ticpp::Exception(msg.str());
			}
		}
    std::cout << "Object setting initial value to '" << init_value << "'" << std::endl;
    setFloatValue(init_value);
  }

  descr_m = pConfig->GetText(false);
  std::cout << "Configured object '" << id_m << "': gad='" << gad_m << "'" << std::endl;
}

void Object::exportXml(ticpp::Element* pConfig)
{
    pConfig->SetAttribute("id", id_m);

    if (gad_m != 0)
        pConfig->SetAttribute("gad", writegaddr(gad_m));

    if (initValue_m != "")
        pConfig->SetAttribute("init", initValue_m);

    if (descr_m != "")
        pConfig->SetText(descr_m);
}

void Object::read()
{
  uint8_t buf[2] = { 0, 0 };
  Services::instance()->getKnxConnection()->write(getGad(), buf, 2);
  readPending_m = true;

	int cnt = 0;
  while (cnt < 100 && readPending_m)
  {
    pth_usleep(10000);
    ++cnt;
  }
}

void Object::onUpdate()
{
    ListenerList_t::iterator it;
    for (it = listenerList_m.begin(); it != listenerList_m.end(); it++)
    {
//        std::cout << "Calling onChange on listener for " << id_m << std::endl;
        (*it)->onChange(this);
    }
    if (initValue_m == "persist")
    {
				PersistentStorage::write(id_m, getFloatValue());
    }
}

void Object::onWrite(const uint8_t* buf, int len)
{
    readPending_m = false;
}

void Object::addChangeListener(ChangeListener* listener)
{
    std::cout << "Adding listener to object '" << id_m << "'" << std::endl;
    listenerList_m.push_back(listener);
}
void Object::removeChangeListener(ChangeListener* listener)
{
    listenerList_m.remove(listener);
}

SwitchingObject::SwitchingObject() : value_m(false)
{
}

SwitchingObject::~SwitchingObject()
{
}

void SwitchingObject::onWrite(const uint8_t* buf, int len)
{
    bool newValue;
    Object::onWrite(buf, len);
    if (len == 2)
        newValue = (buf[1] & 0x3F) != 0;
    else
        newValue = buf[2] != 0;
    if (!init_m || newValue != value_m)
    {
        std::cout << "New value " << newValue << " for switching object " << getID() << std::endl;
        value_m = newValue;
        init_m = true;
        onUpdate();
    }
}

void SwitchingObject::setIntValue(int value)
{
    setBoolValue(value != 0);
}

void SwitchingObject::setFloatValue(float value)
{
    setBoolValue(value != 0);
}

void SwitchingObject::setBoolValue(bool value)
{
    if (!init_m || value != value_m)
    {
        value_m = value;
        uint8_t buf[3] = { 0, 0x80 };
        buf[1] = value ? 0x81 : 0x80;
        Services::instance()->getKnxConnection()->write(getGad(), buf, 2);
        init_m = true;
        onUpdate();
    }
}


ValueObject::ValueObject() : value_m(0)
{
}

ValueObject::~ValueObject()
{
}

void ValueObject::exportXml(ticpp::Element* pConfig)
{
    Object::exportXml(pConfig);
    pConfig->SetAttribute("type", "EIS5");
}

void ValueObject::onWrite(const uint8_t* buf, int len)
{
    if (len < 4)
    {
        std::cout << "Invlalid packet received for ValueObject (too short)" << std::endl;
        return;
    }
    float newValue;
    Object::onWrite(buf, len);
    int d1 = ((unsigned char) buf[2]) * 256 + (unsigned char) buf[3];
    int m = d1 & 0x7ff;
    int ex = (d1 & 0x7800) >> 11;
    newValue = ((float)m * (1 << ex) / 100);
//		printf ("d1=%d;m=%d;ex=%d;temp=%f\n", d1, m, ex, temp);
    if (!init_m || newValue != value_m)
    {
        std::cout << "New value " << newValue << " for value object " << getID() << std::endl;
        value_m = newValue;
        init_m = true;
        onUpdate();
    }
}

void ValueObject::setIntValue(int value)
{
    setFloatValue((float)value);
}

void ValueObject::setFloatValue(float value)
{
    if (value != value_m)
    {
        value_m = value;
        uint8_t buf[4] = { 0, 0x80, 0, 0 };
        int ex = 0;
        int m = (int)(value * 100);
        while (m > 2047 || m < -2048)
        {
            m >> 1;
            ex++;
        }
        if (m < 0)
        {
            m += 2048;
            buf[2] = ((m >> 8) & 0x07) | ((ex << 3) & 0x78) | (1 << 7);
        }
        else
        {
            buf[2] = ((m >> 8) & 0x07) | ((ex << 3) & 0x78);
        }
        buf[3] = (m & 0xff);
        
        Services::instance()->getKnxConnection()->write(getGad(), buf, 4);
        onUpdate();
    }
}

void ValueObject::setBoolValue(bool value)
{
    setFloatValue(value ? 1 : 0);
}



ScalingObject::ScalingObject() : value_m(0)
{
}

ScalingObject::~ScalingObject()
{
}

void ScalingObject::exportXml(ticpp::Element* pConfig)
{
    Object::exportXml(pConfig);
    pConfig->SetAttribute("type", "EIS6");
}

void ScalingObject::onWrite(const uint8_t* buf, int len)
{
    int newValue;
    Object::onWrite(buf, len);
    if (len == 2)
        newValue = (buf[1] & 0x3F);
    else
        newValue = buf[2];
    if (!init_m || newValue != value_m)
    {
        std::cout << "New value " << newValue << " for scaling object " << getID() << std::endl;
        value_m = newValue;
        init_m = true;
        onUpdate();
    }
}

void ScalingObject::setIntValue(int value)
{
    if (value != value_m)
    {
        value_m = value;
        uint8_t buf[3] = { 0, 0x80, 0 };
        buf[2] = (value & 0xff);
        
        Services::instance()->getKnxConnection()->write(getGad(), buf, 3);
        onUpdate();
    }
}

void ScalingObject::setFloatValue(float value)
{
    setIntValue((int)value);
}

void ScalingObject::setBoolValue(bool value)
{
    setIntValue(value ? 1 : 0);
}


ObjectController::ObjectController()
{
    Services::instance()->getKnxConnection()->addTelegramListener(this);
}

ObjectController::~ObjectController()
{
  ObjectIdMap_t::iterator it;
  for (it = objectIdMap_m.begin(); it != objectIdMap_m.end(); it++)
    delete (*it).second;
}

ObjectController* ObjectController::instance()
{
  if (instance_m == 0)
    instance_m = new ObjectController();
  return instance_m;
}

void ObjectController::onWrite(eibaddr_t src, eibaddr_t dest, const uint8_t* buf, int len)
{
        ObjectMap_t::iterator it = objectMap_m.find(dest);
        if (it != objectMap_m.end())
          (*it).second->onWrite(buf, len);
}

void ObjectController::onRead(eibaddr_t src, eibaddr_t dest, const uint8_t* buf, int len) { };
void ObjectController::onResponse(eibaddr_t src, eibaddr_t dest, const uint8_t* buf, int len) { onWrite(src, dest, buf, len); };

Object* ObjectController::getObject(const std::string& id)
{
  ObjectIdMap_t::iterator it = objectIdMap_m.find(id);
  if (it == objectIdMap_m.end())
  {
		std::stringstream msg;
		msg << "ObjectController: Object ID not found: '" << id << "'" << std::endl;
		throw ticpp::Exception(msg.str());
  }
  return (*it).second;
}

void ObjectController::addObject(Object* object)
{
  if (!objectIdMap_m.insert(ObjectIdPair_t(object->getID(), object)).second)
    throw ticpp::Exception("Object ID already exists");
  if (object->getGad() && !objectMap_m.insert(ObjectPair_t(object->getGad(), object)).second)
    throw ticpp::Exception("Object GAD is already registered");
}

void ObjectController::importXml(ticpp::Element* pConfig)
{
  ticpp::Iterator< ticpp::Element > child;
  for ( child = pConfig->FirstChildElement(); child != child.end(); child++ )
  {
    if (child->Value() == "object")
    {
      Object* object = Object::create(&(*child));
      objectIdMap_m[object->getID()] = object;
      eibaddr_t gad = object->getGad();
      if (gad)
        objectMap_m[gad] = object;
    }
  }

}

void ObjectController::exportXml(ticpp::Element* pConfig)
{
  ObjectIdMap_t::iterator it;
  for (it = objectIdMap_m.begin(); it != objectIdMap_m.end(); it++)
  {
    ticpp::Element pElem("object");
    (*it).second->exportXml(&pElem);
    pConfig->LinkEndChild(&pElem);
  }
}
